﻿using Rewired;
using System.Collections;
using UnityEngine;
using UnityEngine.UI;

[RequireComponent(typeof(IPlayerInputComponent))]
public class PlayerController : InteractableObject
{
    #region Variables

    #region GeneralVars
    /// <summary>
    /// The layer mask used for player interaction
    /// </summary>
    public string mask;
    /// <summary>
    /// The layer mask used for friendly bots
    /// </summary>
    public string botMask;
    /// <summary>
    /// The Rigidbody2D attached to this player
    /// </summary>
    private Rigidbody2D rb;

    /// <summary>
    /// Reference to BenScreenShake.cs
    /// </summary>
    public BenScreenShake screenShake;
    /// <summary>
    /// The previously highlighted interactable object.
    /// </summary>
    private InteractableObject previousAttackHighlight = null;

    private InteractableObject previousActionHighlight = null;

    private SoundManager soundManager;
    #endregion

    #region Movement/InteractionVars
    /// <summary>
    /// The speed at which the player moves
    /// </summary>
    [Range(0, 5)] public float moveSpeed;
    [HideInInspector]
    public IPlayerInputComponent inputComponent;

    private string otherPlayer;
    private string otherBot;
    #endregion

    #region StunVars
    /// <summary>
    /// Is this player stunned?
    /// </summary>
    [HideInInspector] public bool isStunned = false;
    /// <summary>
    /// Can the player be stunned? Allows for stunRepeatTime to be greater than 0
    /// </summary>
    private bool canBeStunned = true;
    /// <summary>
    /// The time that the player is stunned when hit
    /// </summary>
    [Range(0, 5)] public float stunTime = 1.5f;
    /// <summary>
    /// The halo that appears when the player is stunned
    /// </summary>
    private GameObject halo;
    /// <summary>
    /// The time after a stun ends during which the player cannot be stunned again
    /// </summary>
    [Range(0, 5)] public float stunRepeatTime = 0;
    #endregion

    #region UpgradeVars
    /// <summary>
    /// Is this player currently holding an upgrade?
    /// </summary>
    [HideInInspector]
    public bool isHoldingUpgrade = false;
    /// <summary>
    /// The upgrade this player is currently holding (if any)
    /// </summary>
    [HideInInspector]
    public GameObject heldUpgrade = null;
    #endregion

    #region ScrapVars
    /// <summary>
    /// The scrap prefab, used to drop scrap when hit
    /// </summary>
    public GameObject scrap;
    /// <summary>
    /// The amount of scrap the player is currently holding
    /// </summary>
    public int scrapNum;
    /// <summary>
    /// The Text object that displays the current scrap number on the robot's chest
    /// </summary>
    public Text scrapText;
    #endregion

    #region MapConstraints
    /// <summary>
    /// The maximum X position that this player can move to
    /// </summary>
    public float maxX = 8.5f;
    /// <summary>
    /// The minimum X position that this player can move to
    /// </summary>
    public float minX = -8.5f;
    /// <summary>
    /// The maximum Y position that this player can move to
    /// </summary>
    private float maxY = 1.3f;
    /// <summary>
    /// The minimum Y position that this player can move to
    /// </summary>
    private float minY = -4f;
    #endregion

    #endregion

    // Start is called before the first frame update
    public void Start()
    {
        type = "Player";
        //base.player = player + 1; // Normalises this player's variable with Devin's not-starting-at-0 numbering scheme for the robots

        if (team == 0)
        {
            SpawnedObjectManager.redPlayers.Add(this);
        }
        else
        {
            SpawnedObjectManager.bluePlayers.Add(this);
        }

        inputComponent = gameObject.GetComponent<IPlayerInputComponent>();

        rb = GetComponent<Rigidbody2D>(); // Assign the rigidbody component for movement

        isHoldingUpgrade = false; // The player isn't holding an upgrade when the game starts
        heldUpgrade = null;

        halo = transform.Find("Halo").gameObject; // Sets the halo object for when the player gets stunned

        isStunned = false;
        canBeStunned = true;

        soundManager = FindObjectOfType<SoundManager>();
        if (team == 0)
        {
            otherPlayer = "Player2";
            otherBot = "Robot2";
        }
        else
        {
            otherPlayer = "Player1";
            otherBot = "Robot1";
        }
    }

    /// <summary>
    /// Called at fixed intervals. Used for movement input handling.
    /// </summary>
    void FixedUpdate()
    {
        if (!isStunned)
        {
            halo.SetActive(false); // Disable the player's stun halo

            ProcessMovementInput(); // Process the movement input to move the player
        }
        else

        {
            halo.SetActive(true); // Enable the player's stun halo
        }
    }

    /// <summary>
    /// Called every frame. Used for action input handling
    /// </summary>
    private void Update()
    {
        if (!isStunned && !PauseState.isPaused)
        {
            ProcessActionInput(); // Process the action input to perform an action
        }

        scrapText.text = scrapNum.ToString(); // Update the scrap text on the player's chest to match their current scrap total

        if (heldUpgrade == null)
        {
            isHoldingUpgrade = false; // Ensures that the player doesn't get stuck holding an upgrade
        }
    }

    /// <summary>
    /// Takes the moveVector updated in GetMovementInput() and uses it to move the player.
    /// </summary>
    private void ProcessMovementInput()
    {
        if ((inputComponent.MoveVector.x != 0.0f || inputComponent.MoveVector.y != 0.0f)) // Checks that player is moving
        {
            rb.velocity = inputComponent.MoveVector * moveSpeed * Time.deltaTime * 100; // Sets the preliminary movement velocity

            // Makes sure the player is in bounds, not exceeding the play area.
            #region BoundCheck 
            if (gameObject.transform.position.x > maxX)
            {
                if (inputComponent.MoveVector.x > 0f)
                {
                    rb.velocity = new Vector2(0, rb.velocity.y);
                }
            }
            if (gameObject.transform.position.x < minX)
            {
                if (inputComponent.MoveVector.x < 0f)
                {
                    rb.velocity = new Vector2(0, rb.velocity.y);
                }
            }
            if (gameObject.transform.position.y > maxY)
            {
                if (inputComponent.MoveVector.y > 0f)
                {
                    rb.velocity = new Vector2(rb.velocity.x, 0);
                }
            }
            if (gameObject.transform.position.y < minY)
            {
                if (inputComponent.MoveVector.y < 0f)
                {
                    rb.velocity = new Vector2(rb.velocity.x, 0);
                }
            }
            #endregion 
        }
        else
        {
            rb.velocity *= 0; // If the input is 0, set the velocity to 0
        }
    }

    /// <summary>
    /// Takes the action bool from the InputComponent and uses it to interact with the world
    /// </summary>
    private void ProcessActionInput()
    {
        #region RayConstruction
        RaycastHit2D attackHit = Physics2D.Raycast(transform.position + new Vector3(0, 0.5f, 0), Vector2.down, 0.85f, ~LayerMask.GetMask(mask, botMask, "BotCollide", "Upgrade", "Ignore Raycast")); // Primary ray, goes through the center of the player. Ignores scrap and upgrades
        RaycastHit2D attackHit1 = Physics2D.Raycast(transform.position + new Vector3(-0.45f, 0.4f, 0), Vector2.down, 0.75f, ~LayerMask.GetMask(mask, botMask, "BotCollide", "Upgrade", "Ignore Raycast")); // Secondary ray, goes through the left side of the player. Ignores scrap and upgrades
        RaycastHit2D attackHit2 = Physics2D.Raycast(transform.position + new Vector3(0.45f, 0.4f, 0), Vector2.down, 0.75f, ~LayerMask.GetMask(mask, botMask, "BotCollide", "Upgrade", "Ignore Raycast")); // Tertiary ray, goes through the right side of the player. Ignores scrap and upgrades

        RaycastHit2D actionHit = Physics2D.Raycast(transform.position + new Vector3(0, 0.5f, 0), Vector2.down, 0.85f, ~LayerMask.GetMask(mask, botMask, "BotCollide", otherPlayer, otherBot, "Robot", "Ignore Raycast")); // Primary ray, goes through the center of the player. Ignores players and robots
        RaycastHit2D actionHit1 = Physics2D.Raycast(transform.position + new Vector3(-0.45f, 0.4f, 0), Vector2.down, 0.75f, ~LayerMask.GetMask(mask, botMask, "BotCollide", otherPlayer, otherBot, "Robot", "Ignore Raycast")); // Secondary ray, goes through the left side of the player. Ignores players and robots
        RaycastHit2D actionHit2 = Physics2D.Raycast(transform.position + new Vector3(0.45f, 0.4f, 0), Vector2.down, 0.75f, ~LayerMask.GetMask(mask, botMask, "BotCollide", otherPlayer, otherBot, "Robot", "Ignore Raycast")); // Tertiary ray, goes through the right side of the player. Ignores players and robots
        #endregion

        #region InteractableHighlight
        InteractableObject attackHighlightObject = null;
        if (attackHit)
            attackHighlightObject = attackHit.collider.GetComponent<InteractableObject>();
        else if (attackHit1)
            attackHighlightObject = attackHit1.collider.GetComponent<InteractableObject>();
        else if (attackHit2)
            attackHighlightObject = attackHit2.collider.GetComponent<InteractableObject>();
        else
            attackHighlightObject = null;

        if (attackHighlightObject != null)
        {
            if (attackHighlightObject.Highlight == -1)
            {
                attackHighlightObject.Highlight = team;
                if (previousAttackHighlight != null && previousAttackHighlight != attackHighlightObject)
                {
                    previousAttackHighlight.Highlight = -1;
                }
                previousAttackHighlight = attackHighlightObject;
            }
        }
        else if (previousAttackHighlight != null)
        {
            previousAttackHighlight.Highlight = -1;
        }
        else if (attackHighlightObject = null)
        {
            previousAttackHighlight.Highlight = -1;
        }

        InteractableObject actionHighlightObject = null;
        if (actionHit)
            actionHighlightObject = actionHit.collider.GetComponent<InteractableObject>();
        else if (actionHit1)
            actionHighlightObject = actionHit1.collider.GetComponent<InteractableObject>();
        else if (actionHit2)
            actionHighlightObject = actionHit2.collider.GetComponent<InteractableObject>();
        else
            actionHighlightObject = null;

        if (actionHighlightObject != null)
        {
            if (actionHighlightObject.Highlight == -1)
            {
                actionHighlightObject.Highlight = team;
                if (previousActionHighlight != null && previousActionHighlight != actionHighlightObject)
                {
                    previousActionHighlight.Highlight = -1;
                }
                previousActionHighlight = actionHighlightObject;
            }
        }
        else if (previousActionHighlight != null)
        {
            previousActionHighlight.Highlight = -1;
        }
        else if (actionHighlightObject = null)
        {
            previousActionHighlight.Highlight = -1;
        }
        #endregion

        if (inputComponent.Attack)
        {
            bool foundObject = false;

            if (attackHit) // Checks if primary ray hits anything
            {
                InteractableObject attackHitObject = attackHit.collider.GetComponent<InteractableObject>();
                if (attackHitObject != null) // If the hit object has an InteractableObject script and the player isn't trying to pick up a second upgrade (already holding one)
                {
                    attackHitObject.Activate(gameObject); // Call the Activate() function on the hit object's InteractableObject script

                    if (attackHitObject.GetType() == "Robot" && attackHitObject.team == team) // If the hitObject is a friendly robot
                    {
                        foundObject = true; // Set the pickup/repair animation to play
                    }
                }
            }
            else if (attackHit1)
            {
                InteractableObject attackHitObject = attackHit1.collider.GetComponent<InteractableObject>();
                if (attackHitObject != null) // If the hit object has an InteractableObject script and the player isn't trying to pick up a second upgrade (already holding one)
                {
                    attackHitObject.Activate(gameObject); // Call the Activate() function on the hit object's InteractableObject script

                    if (attackHitObject.GetType() == "Robot" && attackHitObject.team == base.team /*hitObject.objectType == InteractableObject.TypeSelector.robotRed*/) // If the hitObject is a friendly robot
                    {
                        foundObject = true; // Set the pickup/repair animation to play
                    }
                }
            }
            else if (attackHit2)
            {
                InteractableObject attackHitObject = attackHit2.collider.GetComponent<InteractableObject>();
                if (attackHitObject != null) // If the hit object has an InteractableObject script and the player isn't trying to pick up a second upgrade (already holding one)
                {
                    attackHitObject.Activate(gameObject); // Call the Activate() function on the hit object's InteractableObject script

                    if (attackHitObject.GetType() == "Robot" && attackHitObject.team == base.team /*hitObject.objectType == InteractableObject.TypeSelector.robotRed*/) // If the hitObject is a friendly robot
                    {
                        foundObject = true; // Set the pickup/repair animation to play
                    }
                }
            }

            if (foundObject) // If the action repaired something
            {
                PlayPickup(); // Play the pickup or repair animation
            }
            else
            {
                PlayAttack(); // Play the attack animation
            }
        }

        else if (inputComponent.Action) // Updated in GetActionInput()
        {
            if (actionHit) // Checks if primary ray hits anything
            {
                InteractableObject hitObject = actionHit.collider.GetComponent<InteractableObject>();
                if (hitObject != null && (!isHoldingUpgrade || hitObject.GetType() == "Scrap" || hitObject.GetType() == "Robot")) // If the hit object has an InteractableObject script and the player isn't trying to pick up a second upgrade (already holding one)
                {
                    hitObject.Activate(gameObject); // Call the Activate() function on the hit object's InteractableObject script
                }
            }
            else if (actionHit1) // If primary ray hits nothing, check secondary ray
            {
                InteractableObject hitObject = actionHit1.collider.GetComponent<InteractableObject>();
                if (hitObject != null && (!isHoldingUpgrade || hitObject.GetType() == "Scrap" || hitObject.GetType() == "Robot")) // If the hit object has an InteractableObject script and the player isn't trying to pick up a second upgrade (already holding one)
                {
                    hitObject.Activate(gameObject); // Call the Activate() function on the hit object's InteractableObject script
                }

            }
            else if (actionHit2) // If primary and secondary ray hit nothing, check tertiary ray
            {
                InteractableObject hitObject = actionHit2.collider.GetComponent<InteractableObject>();
                if (hitObject != null && (!isHoldingUpgrade || hitObject.GetType() == "Scrap" || hitObject.GetType() == "Robot")) // If the hit object has an InteractableObject script and the player isn't trying to pick up a second upgrade (already holding one)
                {
                    hitObject.Activate(gameObject); // Call the Activate() function on the hit object's InteractableObject script
                }

            }
            else if (heldUpgrade != null) // If no rays hit anything, check if the player is holding an upgrade. If so, either use it to repair a robot or drop it.
            {
                heldUpgrade.GetComponent<InteractableObject>().Activate(gameObject); // Activate the held upgrade. In this state this will either repair a robot with it or drop it.
            }

            PlayPickup();
        }
    }

    public override void Activate(GameObject source)
    {
        PlayerController controller = source.GetComponent<PlayerController>();

        gameObject.GetComponent<PlayerController>().StartCoroutine("Stun");
        controller.PlayAttack();
    }

    /// <summary>
    /// Prevents the player from moving or interacting with the world for stunTime seconds.
    /// </summary>
    /// <returns></returns>
    public IEnumerator Stun()
    {
        if (canBeStunned) // Checks if the player can be stunned. Player cannot be stunned if they are already stunned, or if they were stunned less than stunRepeatTime seconds ago.
        {
            // Calls screenshake script when player is stunned
            if (screenShake != null)
                screenShake.Shake(.22f, .05f);
            //Debug.Log("screen shake");

            canBeStunned = false; // Signals that the player cannot be stunned again until the end of this coroutine
            isStunned = true; // Tells the movement/interaction systems that the player is stunned
            if (team == 0)
            {
                SpawnedObjectManager.redPlayers.Remove(this);
            }
            else
            {
                SpawnedObjectManager.bluePlayers.Remove(this);
            }
            if (previousActionHighlight != null)
            {
                previousActionHighlight.Highlight = -1;
                previousActionHighlight = null;
            }
            if (previousAttackHighlight != null)
            {
                previousAttackHighlight.Highlight = -1;
                previousAttackHighlight = null;
            }


            //play stun sound
            if ((Random.Range(0f, 1f) < 0.5f))
            {
                try
                {
                    soundManager.Play("wrenchHit1");
                }
                catch (System.NullReferenceException)
                {

                    Debug.Log("No sound manager");
                }
            }
            else if (soundManager)
            {
                try
                {
                    soundManager.Play("wrenchHit2");
                }
                catch (System.NullReferenceException)
                {

                    Debug.Log("No sound manager");
                }
            }

            rb.velocity *= 0; // Sets the velocity to 0
            if (isHoldingUpgrade) // Drops any upgrade that the player is holding
            {
                heldUpgrade.GetComponent<Upgrade>().DropUpgrade(this.gameObject);
            }

            if (scrapNum > 0) // Checks if the player is holding scrap. If so, drops half
            {
                int droppedAmount = Mathf.RoundToInt(scrapNum / 2);
                scrapNum -= droppedAmount;

                //play scrap drop sound
                try
                {
                    soundManager.Play("scrapDrop");
                }
                catch (System.NullReferenceException)
                {

                    Debug.Log("No sound manager");
                }

                if (droppedAmount != 0) // Checks if the player dropped scrap (if they had more than 1 in the first place)
                {
                    for (int i = 0; i < droppedAmount; i++) // Places the dropped scrap at random positions around the player
                    {
                        if (Ben_Pickup_Spawn.numSpawnedScrap < Ben_Pickup_Spawn.maxSpawnedScrap)
                        {
                            GameObject newScrap = Instantiate(scrap);
                            newScrap.transform.position = gameObject.transform.position + new Vector3(Random.Range(-2f, 2f), Random.Range(-.5f, .5f));
                        }
                        else
                        {
                            break;
                        }
                    }
                }
            }

            yield return new WaitForSeconds(stunTime); // Waits for stunTime seconds before re-enabling movement/interaction

            isStunned = false; // Tells the movement/interaciton systems that the player is no longer stunned

            yield return new WaitForSeconds(stunRepeatTime); // The player is not stunned, but cannot be stunned again
            canBeStunned = true; // Allows the player to be stunned again

            if (team == 0)
            {
                SpawnedObjectManager.redPlayers.Add(this);
            }
            else
            {
                SpawnedObjectManager.bluePlayers.Add(this);
            }
        }
    }

    /// <summary>
    /// Plays the attack animation for the player
    /// </summary>
    public void PlayAttack()
    {
        if (team == 0)
        {
            GetComponent<Animator>().Play("Player1Attack");

            //play wrench swing sound
            try
            {
                soundManager.Play("wrenchSwing");
            }
            catch (System.NullReferenceException)
            {

                Debug.Log("No sound manager");
            }
        }
        else if (team == 1)
        {
            GetComponent<Animator>().Play("Player2Attack");

            //play wrench swing sound
            try
            {
                soundManager.Play("wrenchSwing");
            }
            catch (System.NullReferenceException)
            {

                Debug.Log("No sound manager");
            }
        }
    }

    /// <summary>
    /// Plays the Pickup animation for the player
    /// </summary>
    public void PlayPickup()
    {
        if (team == 0)
        {
            GetComponent<Animator>().Play("Player1Pickup");

            //play scrap pickup sound
            try
            {
                soundManager.Play("scrapPickup");
            }
            catch (System.NullReferenceException)
            {

                Debug.Log("No sound manager");
            }

        }
        else if (team == 1)
        {
            GetComponent<Animator>().Play("Player2Pickup");

            //play scrap pickup sound
            try
            {
                soundManager.Play("scrapPickup");
            }
            catch (System.NullReferenceException)
            {

                Debug.Log("No sound manager");
            }
        }
    }
}
