﻿using System.Collections;
using System.Collections.Generic;
using UnityEngine;

public class AIInputComponent : MonoBehaviour, IPlayerInputComponent
{
    // True on the frame that Action is pressed.
    public bool Action { get; private set; }
    // True on the frame that Attack is pressed
    public bool Attack { get; private set; }
    // Vector of the x and y input components
    public Vector3 MoveVector { get; private set; }

    // If true, this bot should target the nearest of any object. If false, target the farthest away.
    public bool nearest = true;

    // The playercontroller on this gameobject. Send input signals to this.
    private PlayerController controller;

    // The current target transform, used to target movement.
    private Transform target;

    // State structure for this AI
    private enum State
    {
        Seek,
        MoveAndAction,
        MoveAndAttack,
        GetUpgrade,
        GetScrap,
        ActivateBot,
        StunBot,
        StunPlayer
    }
    private State playerState = State.Seek;

    void Start()
    {
        controller = gameObject.GetComponent<PlayerController>();
        StartCoroutine(nameof(CheckPriority));
    }

    void Update()
    {
        switch (playerState)
        {
            case State.Seek:
                SeekState();
                break;
            case State.MoveAndAction:
                if (MoveTowardsObject() < 0.2f)
                {
                    StartCoroutine(nameof(SendAction));
                    playerState = State.Seek;
                }
                break;
            case State.MoveAndAttack:
                if (MoveTowardsObject() < 0.2f)
                {
                    StartCoroutine(nameof(SendAttack));
                    playerState = State.Seek;
                }
                break;
            case State.GetUpgrade:
                if (SpawnedObjectManager.spawnedUpgrades.Contains(target.gameObject.GetComponent<Upgrade>()))
                    goto case State.MoveAndAction;
                playerState = State.Seek;
                break;
            case State.GetScrap:
                goto case State.MoveAndAction;
            case State.ActivateBot:
                goto case State.MoveAndAction;
            case State.StunBot:
                goto case State.MoveAndAttack;
            case State.StunPlayer:
                goto case State.MoveAndAttack;
        }
    }

    /// <Summary>
    /// Finds the hightest-priority available target transform and sets target to that transform. Moves to other states based on what it finds.
    /// </Summary>
    private void SeekState()
    {
        InteractableObject targetScript = null;
        if (!controller.isHoldingUpgrade) // If bot is not holding an upgrade, try to find one. If there are none available, move on.
        {
            targetScript = nearest ? SpawnedObjectManager.GetNearest(transform.position, SpawnedObjectManager.spawnedUpgrades) : SpawnedObjectManager.GetFarthest(transform.position, SpawnedObjectManager.spawnedUpgrades);
            if (targetScript != null)
            {
                target = targetScript.transform;
                playerState = State.GetUpgrade;
                return;
            }
        }

        // Player is holding an upgrade or there are none available.
        if (controller.isHoldingUpgrade || controller.scrapNum > 0) // If player has an upgrade or scrap, try to activate a bot. If there are none available, move on.
        {
            targetScript = nearest ? SpawnedObjectManager.GetNearest(transform.position, controller.team == 0 ? SpawnedObjectManager.redIdleBots : SpawnedObjectManager.blueIdleBots) : SpawnedObjectManager.GetFarthest(transform.position, controller.team == 0 ? SpawnedObjectManager.redIdleBots : SpawnedObjectManager.blueIdleBots);
            if (targetScript != null)
            {
                target = targetScript.transform;
                playerState = State.ActivateBot;
                return;
            }
        }

        // Player has no scrap/upgrades, or there are no free bots to activate.
        if (controller.scrapNum < 6) // If bot has less than 6 scrap, try to find some more. If there are none avaiable, move on.
        {
            targetScript = nearest ? SpawnedObjectManager.GetNearest(transform.position, SpawnedObjectManager.spawnedScrap) : SpawnedObjectManager.GetFarthest(transform.position, SpawnedObjectManager.spawnedScrap);
            if (targetScript != null)
            {
                target = targetScript.transform;
                playerState = State.GetScrap;
                return;
            }
        }

        // There are no free bots to activate and the player has at least 6 scrap.
        targetScript = nearest ? SpawnedObjectManager.GetNearest(transform.position, controller.team == 0 ? SpawnedObjectManager.blueActiveBots : SpawnedObjectManager.redActiveBots) : SpawnedObjectManager.GetFarthest(transform.position, controller.team == 0 ? SpawnedObjectManager.blueActiveBots : SpawnedObjectManager.redActiveBots); // If there's an an enemy bot available to stun, stun it.
        if (targetScript != null)
        {
            target = targetScript.transform;
            playerState = State.StunBot;
            return;
        }

        // There are no free bots to activate, the player has at least 6 scrap, and there are no bots to stun. Try to stun player.
        targetScript = nearest ? SpawnedObjectManager.GetNearest(transform.position, controller.team == 0 ? SpawnedObjectManager.bluePlayers : SpawnedObjectManager.redPlayers) : SpawnedObjectManager.GetFarthest(transform.position, controller.team == 0 ? SpawnedObjectManager.bluePlayers : SpawnedObjectManager.redPlayers);
        if (targetScript != null)
        {
            target = targetScript.transform;
            playerState = State.StunPlayer;
            return;
        }

        // There are no free bots, to activate, the player has at least 6 scrap, and there are no players or bots to stun. Stand still and keep seeking.
        target = transform;
        MoveVector = Vector3.zero;
        playerState = State.Seek;
    }

    /// <Summary>
    /// Moves towards the target object, returns the distance to the object.
    /// </Summary>
    private float MoveTowardsObject() // Move towards the target object.
    {
        // If there is a target object, set the MoveVector towards it. Return the distance.
        if (target != null)
        {
            Vector3 direction = target.position - transform.position;
            float distance = direction.magnitude;
            MoveVector = direction / distance;
            return distance;
        }
        // If there is no target object (object destroyed/missing), reset to the Seek state and return Infinity.
        playerState = State.Seek;
        return Mathf.Infinity;
    }
    /// <Summary>
    /// Simulates action input.
    /// </Summary>
    private IEnumerator SendAction()
    {
        Action = true;
        yield return new WaitForEndOfFrame();
        Action = false;
    }

    /// <Summary>
    /// Simulates attack input.
    /// </Summary>
    private IEnumerator SendAttack()
    {
        Attack = true;
        yield return new WaitForEndOfFrame();
        Attack = false;
    }

    /// <Summary>
    /// Reverts to seek state every few seconds to make sure the AI doesn't get stuck chasing a player/robot
    /// </Summary>
    private IEnumerator CheckPriority()
    {
        playerState = State.Seek;
        yield return new WaitForSeconds(5);
        StartCoroutine(nameof(CheckPriority));
    }
}
